<?php
/**
 * Get domain creation date (tries RDAP, then whois fallback).
 * Returns JSON response with domain information.
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');

function get_domain_creation_date(string $domain) {
    $domain = trim(strtolower($domain));
    // remove protocol and path if user passed full URL
    $domain = preg_replace('#^https?://#', '', $domain);
    $domain = preg_replace('#/.*$#', '', $domain);

    // convert IDN (unicode) domains to punycode if needed
    if (function_exists('idn_to_ascii')) {
        $puny = idn_to_ascii($domain, IDNA_DEFAULT, INTL_IDNA_VARIANT_UTS46);
        if ($puny !== false) $domain = $puny;
    }

    // --- 1) Try RDAP (JSON)
    $rdap_urls = [
        "https://rdap.org/domain/{$domain}",
        "https://rdap.verisign.com/com/v1/domain/{$domain}"
    ];
    foreach ($rdap_urls as $url) {
        $json = curl_get($url);
        if ($json) {
            $data = json_decode($json, true);
            if (is_array($data)) {
                // RDAP common place for events: 'events' or 'eventAction' array varies
                if (!empty($data['events']) && is_array($data['events'])) {
                    foreach ($data['events'] as $ev) {
                        if (!empty($ev['eventAction']) && stripos($ev['eventAction'], 'reg') !== false) {
                            if (!empty($ev['eventDate'])) {
                                $date = try_parse_date($ev['eventDate']);
                                if ($date) {
                                    return [
                                        'success' => true,
                                        'created_date' => $date,
                                        'registrar' => $data['entities'][0]['vcardArray'][1][1][3] ?? 'Unknown',
                                        'registry' => 'RDAP',
                                        'nameservers' => isset($data['nameservers']) ? array_column($data['nameservers'], 'ldhName') : [],
                                        'domain_id' => $data['handle'] ?? 'Unknown'
                                    ];
                                }
                            }
                        }
                    }
                }
                // some RDAP responses have 'registration' fields or 'created'
                if (!empty($data['registration']) && !empty($data['registration']['date'])) {
                    $date = try_parse_date($data['registration']['date']);
                    if ($date) {
                        return [
                            'success' => true,
                            'created_date' => $date,
                            'registrar' => $data['entities'][0]['vcardArray'][1][1][3] ?? 'Unknown',
                            'registry' => 'RDAP',
                            'nameservers' => isset($data['nameservers']) ? array_column($data['nameservers'], 'ldhName') : [],
                            'domain_id' => $data['handle'] ?? 'Unknown'
                        ];
                    }
                }
            }
        }
    }

    // --- 2) Fallback to WHOIS (system call)
    $whois_raw = whois_lookup($domain);
    if ($whois_raw) {
        // common patterns for creation/registered date in WHOIS outputs
        $patterns = [
            '/Creation Date:\s*(.+)/i',
            '/Created On:\s*(.+)/i',
            '/Registered On:\s*(.+)/i',
            '/Registered:\s*(.+)/i',
            '/Domain Registration Date:\s*(.+)/i',
            '/Created:\s*(.+)/i',
            '/created:\s*(.+)/i',
            '/registration date:\s*(.+)/i',
            '/registered:\s*(.+)/i'
        ];
        foreach ($patterns as $pat) {
            if (preg_match($pat, $whois_raw, $m)) {
                $date = try_parse_date(trim($m[1]));
                if ($date) {
                    // Try to extract more info from WHOIS
                    $registrar = 'Unknown';
                    if (preg_match('/Registrar:\s*(.+)/i', $whois_raw, $reg_match)) {
                        $registrar = trim($reg_match[1]);
                    }

                    $nameservers = [];
                    if (preg_match_all('/Name Server:\s*(.+)/i', $whois_raw, $ns_matches)) {
                        $nameservers = array_map('trim', $ns_matches[1]);
                    }

                    return [
                        'success' => true,
                        'created_date' => $date,
                        'registrar' => $registrar,
                        'registry' => 'WHOIS',
                        'nameservers' => $nameservers,
                        'domain_id' => 'Unknown'
                    ];
                }
            }
        }
        // try more generic ISO datetime search
        if (preg_match('/(\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}Z)/', $whois_raw, $m)) {
            $date = try_parse_date($m[1]);
            if ($date) {
                return [
                    'success' => true,
                    'created_date' => $date,
                    'registrar' => 'Unknown',
                    'registry' => 'WHOIS',
                    'nameservers' => [],
                    'domain_id' => 'Unknown'
                ];
            }
        }
    }

    // --- 3) If all fails, return failure
    return ['success' => false, 'error' => 'Domain creation date not found'];
}

function curl_get($url, $timeout = 8) {
    $ch = curl_init($url);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_FOLLOWLOCATION => true,
        CURLOPT_CONNECTTIMEOUT => 5,
        CURLOPT_TIMEOUT => $timeout,
        CURLOPT_USERAGENT => 'DomainAgeChecker/1.0 (+loganixseo.com)'
    ]);
    $res = curl_exec($ch);
    $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    return ($res && $code >= 200 && $code < 300) ? $res : false;
}

function whois_lookup($domain) {
    // prefer system whois if available
    if (function_exists('shell_exec')) {
        $cmd = 'whois ' . escapeshellarg($domain) . ' 2>&1';
        $out = @shell_exec($cmd);
        if ($out && strlen(trim($out)) > 0) return $out;
    }
    // fallback: simple socket connection to whois server
    $tlds = explode('.', $domain);
    $tld = array_pop($tlds);
    $whoisServer = 'whois.iana.org';

    $fp = @fsockopen($whoisServer, 43, $errno, $errstr, 5);
    if (!$fp) return false;
    fputs($fp, $domain . "\r\n");
    $out = '';
    while (!feof($fp)) $out .= fgets($fp, 128);
    fclose($fp);
    return $out;
}

function try_parse_date($raw) {
    $raw = trim($raw);
    // remove timezone WORD like 'GMT' or trailing text
    $raw = preg_replace('/\s+\(.*\)$/', '', $raw);
    // common formats: 2020-01-15, 2018-04-10T12:34:56Z, 10-Apr-2012, Apr 10 2012
    $try = strtotime($raw);
    if ($try !== false) {
        // return date only (ISO 8601)
        return date('Y-m-d', $try);
    }
    return false;
}

// Main execution
if (isset($_GET['domain'])) {
    $domain = $_GET['domain'];
    $result = get_domain_creation_date($domain);
    echo json_encode($result);
} else {
    echo json_encode(['success' => false, 'error' => 'No domain specified']);
}
?>